const sqlite3 = require('sqlite3').verbose();
const path = require('path');
const bcrypt = require('bcryptjs');

const dbPath = path.resolve(__dirname, 'database.sqlite');
const db = new sqlite3.Database(dbPath, (err) => {
    if (err) {
        console.error('Error opening database', err.message);
    } else {
        console.log('Connected to the SQLite database.');
        initDb();
    }
});

const initDb = () => {
    db.serialize(() => {
        // Roles Table
        db.run(`CREATE TABLE IF NOT EXISTS roles (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT UNIQUE,
            description TEXT
        )`);

        // Users Table
        db.run(`CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT,
            email TEXT UNIQUE,
            password TEXT,
            role TEXT
        )`);

        // Products Table
        db.run(`CREATE TABLE IF NOT EXISTS products (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT,
            price REAL,
            oldPrice REAL,
            image TEXT,
            category TEXT,
            description TEXT,
            specifications TEXT,
            dimensions TEXT,
            weight REAL,
            materials TEXT,
            care_instructions TEXT,
            warranty_info TEXT,
            sku TEXT UNIQUE,
            barcode TEXT,
            stock INTEGER DEFAULT 0,
            pricing_tier TEXT
        )`);

        // Categories Table
        db.run(`CREATE TABLE IF NOT EXISTS categories (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            title TEXT,
            image TEXT,
            link TEXT
        )`);

        // Hero Table
        db.run(`CREATE TABLE IF NOT EXISTS hero (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            title TEXT,
            subtitle TEXT,
            image TEXT,
            buttonText TEXT,
            active INTEGER DEFAULT 0
        )`);

        // Orders Table
        db.run(`CREATE TABLE IF NOT EXISTS orders (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            total_amount REAL,
            status TEXT DEFAULT 'pending',
            shipping_address TEXT,
            payment_method TEXT,
            tracking_number TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users (id)
        )`);

        // Order Items Table
        db.run(`CREATE TABLE IF NOT EXISTS order_items (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            order_id INTEGER,
            product_id INTEGER,
            quantity INTEGER,
            price REAL,
            FOREIGN KEY (order_id) REFERENCES orders (id),
            FOREIGN KEY (product_id) REFERENCES products (id)
        )`);

        // Seed Roles
        const roles = [
            { name: 'admin', description: 'Administrator with full access' },
            { name: 'customer', description: 'Standard customer user' }
        ];
        
        roles.forEach(role => {
            db.run("INSERT OR IGNORE INTO roles (name, description) VALUES (?, ?)", [role.name, role.description]);
        });

        // Seed Admin User
        db.get("SELECT * FROM users WHERE email = ?", ['admin@arioza.shop'], async (err, row) => {
            if (!row) {
                const hashedPassword = await bcrypt.hash('admin123', 10);
                db.run("INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)", 
                    ['Admin User', 'admin@arioza.shop', hashedPassword, 'admin']);
                console.log('Admin user seeded');
            }
        });

        // Seed Products if empty
        db.get("SELECT count(*) as count FROM products", (err, row) => {
            if (row.count === 0) {
                const products = [
                    {
                        name: "Shoulder Top",
                        price: 12.00,
                        image: "https://images.unsplash.com/photo-1516762689617-e1cffcef479d?w=500&auto=format&fit=crop&q=60",
                        category: "Women"
                    },
                    {
                        name: "Multicolor Shirts",
                        price: 9.00,
                        image: "https://images.unsplash.com/photo-1596755094514-f87e34085b2c?w=500&auto=format&fit=crop&q=60",
                        category: "Men"
                    },
                    {
                        name: "Adidas T-Shirts",
                        price: 12.00,
                        image: "https://images.unsplash.com/photo-1521572163474-6864f9cf17ab?w=500&auto=format&fit=crop&q=60",
                        category: "Men"
                    },
                    {
                        name: "Round-Neck T-Shirts",
                        price: 9.00,
                        image: "https://images.unsplash.com/photo-1583743814966-8936f5b7be1a?w=500&auto=format&fit=crop&q=60",
                        category: "Women"
                    },
                    {
                        name: "Kids Cap",
                        price: 9.00,
                        image: "https://images.unsplash.com/photo-1588850561407-ed78c282e89b?w=500&auto=format&fit=crop&q=60",
                        category: "Kids"
                    },
                    {
                        name: "Jeans Jackets",
                        price: 18.00,
                        image: "https://images.unsplash.com/photo-1576995853123-5a10305d93c0?w=500&auto=format&fit=crop&q=60",
                        category: "Men"
                    },
                    {
                        name: "Office Bag",
                        price: 54.00,
                        oldPrice: 11.23,
                        image: "https://images.unsplash.com/photo-1591561954557-26941169b49e?w=500&auto=format&fit=crop&q=60",
                        category: "Accessories"
                    },
                    {
                        name: "Casual Shoes",
                        price: 24.65,
                        oldPrice: 32.00,
                        image: "https://images.unsplash.com/photo-1560769629-975e13f0c470?w=500&auto=format&fit=crop&q=60",
                        category: "Shoes"
                    }
                ];
                
                const stmt = db.prepare("INSERT INTO products (name, price, oldPrice, image, category) VALUES (?, ?, ?, ?, ?)");
                products.forEach(p => {
                    stmt.run(p.name, p.price, p.oldPrice || null, p.image, p.category);
                });
                stmt.finalize();
                console.log('Products seeded');
            }
        });

        // Seed Categories if empty
        db.get("SELECT count(*) as count FROM categories", (err, row) => {
            if (row.count === 0) {
                const categories = [
                    {
                        title: "DESIGNER BAGS",
                        image: "https://images.unsplash.com/photo-1584917865442-de89df76afd3?w=500&auto=format&fit=crop&q=60",
                        link: "/bags"
                    },
                    {
                        title: "BRANDED WATCH",
                        image: "https://images.unsplash.com/photo-1524592094714-0f0654e20314?w=500&auto=format&fit=crop&q=60",
                        link: "/watches"
                    },
                    {
                        title: "CASUAL SHOES",
                        image: "https://images.unsplash.com/photo-1542291026-7eec264c27ff?w=500&auto=format&fit=crop&q=60",
                        link: "/shoes"
                    }
                ];
                
                const stmt = db.prepare("INSERT INTO categories (title, image, link) VALUES (?, ?, ?)");
                categories.forEach(c => {
                    stmt.run(c.title, c.image, c.link);
                });
                stmt.finalize();
                console.log('Categories seeded');
            }
        });

        // Seed Hero if empty
        db.get("SELECT count(*) as count FROM hero", (err, row) => {
            if (row.count === 0) {
                const hero = {
                    title: "TRENDY SKIRTS",
                    subtitle: "Up to 50% off on top brands",
                    image: "https://images.unsplash.com/photo-1496747611176-843222e1e57c?w=1600&auto=format&fit=crop&q=80",
                    buttonText: "Shop Now",
                    active: 1
                };
                
                db.run("INSERT INTO hero (title, subtitle, image, buttonText, active) VALUES (?, ?, ?, ?, ?)",
                    [hero.title, hero.subtitle, hero.image, hero.buttonText, hero.active]);
                console.log('Hero seeded');
            }
        });
    });
};

module.exports = db;
