const express = require('express');
const cors = require('cors');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const db = require('./database');
const multer = require('multer');
const path = require('path');
const fs = require('fs');

const app = express();
const port = process.env.PORT || 5001;
const SECRET_KEY = process.env.SECRET_KEY || 'your_super_secret_key_change_in_production';

app.use(cors());
app.use(express.json());
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// Serve static files from React app in production
if (process.env.NODE_ENV === 'production') {
    app.use(express.static(path.join(__dirname, 'public')));
}

// Configure Multer
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, path.join(__dirname, 'uploads'));
    },
    filename: (req, file, cb) => {
        cb(null, Date.now() + path.extname(file.originalname));
    }
});

const upload = multer({ storage: storage });

// Middleware
const authenticateToken = (req, res, next) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) return res.sendStatus(401);

    jwt.verify(token, SECRET_KEY, (err, user) => {
        if (err) return res.sendStatus(403);
        req.user = user;
        next();
    });
};

const authorizeAdmin = (req, res, next) => {
    if (req.user.role !== 'admin') {
        return res.status(403).json({ message: 'Access denied: Admins only' });
    }
    next();
};

// Routes

// Upload Endpoint
app.post('/api/upload', upload.single('image'), (req, res) => {
    if (!req.file) {
        return res.status(400).json({ message: 'No file uploaded' });
    }
    const imageUrl = `http://localhost:${port}/uploads/${req.file.filename}`;
    res.json({ imageUrl });
});

// Auth
app.post('/api/register', async (req, res) => {
    const { name, email, password } = req.body;
    
    db.get("SELECT * FROM users WHERE email = ?", [email], async (err, row) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        if (row) return res.status(400).json({ message: 'User already exists' });

        try {
            const hashedPassword = await bcrypt.hash(password, 10);
            db.run("INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)", 
                [name, email, hashedPassword, 'user'], function(err) {
                    if (err) return res.status(500).json({ message: 'Error registering user' });
                    res.status(201).json({ message: 'User registered successfully' });
                });
        } catch (error) {
            res.status(500).json({ message: 'Error registering user' });
        }
    });
});

app.post('/api/login', (req, res) => {
    const { email, password } = req.body;
    
    db.get("SELECT * FROM users WHERE email = ?", [email], async (err, user) => {
        if (err) return res.status(500).send();
        if (!user) return res.status(400).json({ message: 'Cannot find user' });

        try {
            if (await bcrypt.compare(password, user.password)) {
                const accessToken = jwt.sign({ id: user.id, email: user.email, role: user.role, name: user.name }, SECRET_KEY);
                res.json({ accessToken, user: { id: user.id, name: user.name, email: user.email, role: user.role } });
            } else {
                res.status(401).json({ message: 'Not Allowed' });
            }
        } catch {
            res.status(500).send();
        }
    });
});

// Products
app.get('/api/products', (req, res) => {
    const { category } = req.query;
    let query = "SELECT * FROM products";
    let params = [];

    if (category) {
        query += " WHERE category = ?";
        params.push(category);
    }

    db.all(query, params, (err, rows) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        res.json(rows);
    });
});

app.get('/api/products/:id', (req, res) => {
    const id = parseInt(req.params.id);
    db.get("SELECT * FROM products WHERE id = ?", [id], (err, row) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        if (!row) return res.status(404).json({ message: 'Product not found' });
        res.json(row);
    });
});

app.post('/api/products', authenticateToken, authorizeAdmin, (req, res) => {
    const { 
        name, price, oldPrice, image, category, 
        description, specifications, dimensions, weight, 
        materials, care_instructions, warranty_info, sku, 
        barcode, stock, pricing_tier, sizes, colors, images
    } = req.body;

    const sql = `INSERT INTO products (
        name, price, oldPrice, image, category, 
        description, specifications, dimensions, weight, 
        materials, care_instructions, warranty_info, sku, 
        barcode, stock, pricing_tier, sizes, colors, images
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;

    const params = [
        name, price, oldPrice || null, image, category,
        description, specifications, dimensions, weight,
        materials, care_instructions, warranty_info, sku,
        barcode, stock || 0, pricing_tier, sizes, colors, images
    ];

    db.run(sql, params, function(err) {
        if (err) return res.status(500).json({ message: 'Error adding product: ' + err.message });
        res.status(201).json({ id: this.lastID, ...req.body });
    });
});

app.put('/api/products/:id', authenticateToken, authorizeAdmin, (req, res) => {
    const id = parseInt(req.params.id);
    const { 
        name, price, oldPrice, image, category, 
        description, specifications, dimensions, weight, 
        materials, care_instructions, warranty_info, sku, 
        barcode, stock, pricing_tier, sizes, colors, images
    } = req.body;
    
    const sql = `UPDATE products SET 
        name = COALESCE(?, name), 
        price = COALESCE(?, price), 
        oldPrice = COALESCE(?, oldPrice), 
        image = COALESCE(?, image), 
        category = COALESCE(?, category),
        description = COALESCE(?, description),
        specifications = COALESCE(?, specifications),
        dimensions = COALESCE(?, dimensions),
        weight = COALESCE(?, weight),
        materials = COALESCE(?, materials),
        care_instructions = COALESCE(?, care_instructions),
        warranty_info = COALESCE(?, warranty_info),
        sku = COALESCE(?, sku),
        barcode = COALESCE(?, barcode),
        stock = COALESCE(?, stock),
        pricing_tier = COALESCE(?, pricing_tier),
        sizes = COALESCE(?, sizes),
        colors = COALESCE(?, colors),
        images = COALESCE(?, images)
        WHERE id = ?`;

    const params = [
        name, price, oldPrice, image, category,
        description, specifications, dimensions, weight,
        materials, care_instructions, warranty_info, sku,
        barcode, stock, pricing_tier, sizes, colors, images, id
    ];

    db.run(sql, params, function(err) {
        if (err) return res.status(500).json({ message: 'Error updating product: ' + err.message });
        if (this.changes === 0) return res.status(404).json({ message: 'Product not found' });
        res.json({ id, ...req.body });
    });
});

app.delete('/api/products/:id', authenticateToken, authorizeAdmin, (req, res) => {
    const id = parseInt(req.params.id);
    db.run("DELETE FROM products WHERE id = ?", [id], function(err) {
        if (err) return res.status(500).json({ message: 'Error deleting product' });
        if (this.changes === 0) return res.status(404).json({ message: 'Product not found' });
        res.json({ message: 'Product deleted' });
    });
});

// Categories
app.get('/api/categories/inventory', (req, res) => {
    // Get distinct categories with count and one image
    const sql = `
        SELECT 
            category as title, 
            COUNT(*) as count,
            (SELECT image FROM products p2 WHERE p2.category = p1.category LIMIT 1) as image
        FROM products p1 
        GROUP BY category
    `;
    db.all(sql, (err, rows) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        res.json(rows);
    });
});

app.get('/api/categories', (req, res) => {
    db.all("SELECT * FROM categories", (err, rows) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        res.json(rows);
    });
});

app.post('/api/categories', authenticateToken, authorizeAdmin, (req, res) => {
    const { title, image, link } = req.body;
    db.run("INSERT INTO categories (title, image, link) VALUES (?, ?, ?)",
        [title, image, link], function(err) {
            if (err) return res.status(500).json({ message: 'Error adding category' });
            res.status(201).json({ id: this.lastID, ...req.body });
        });
});

app.put('/api/categories/:id', authenticateToken, authorizeAdmin, (req, res) => {
    const id = parseInt(req.params.id);
    const { title, image, link } = req.body;
    db.run(`UPDATE categories SET title = COALESCE(?, title), image = COALESCE(?, image), link = COALESCE(?, link) WHERE id = ?`,
        [title, image, link, id], function(err) {
            if (err) return res.status(500).json({ message: 'Error updating category' });
            if (this.changes === 0) return res.status(404).json({ message: 'Category not found' });
            res.json({ id, ...req.body });
        });
});

app.delete('/api/categories/:id', authenticateToken, authorizeAdmin, (req, res) => {
    const id = parseInt(req.params.id);
    db.run("DELETE FROM categories WHERE id = ?", [id], function(err) {
        if (err) return res.status(500).json({ message: 'Error deleting category' });
        if (this.changes === 0) return res.status(404).json({ message: 'Category not found' });
        res.json({ message: 'Category deleted' });
    });
});

// Hero Content
app.get('/api/hero', (req, res) => {
    db.all("SELECT * FROM hero", (err, rows) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        res.json(rows);
    });
});

app.get('/api/hero/active', (req, res) => {
    db.get("SELECT * FROM hero WHERE active = 1", (err, row) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        res.json(row || {});
    });
});

app.post('/api/hero', authenticateToken, authorizeAdmin, (req, res) => {
    const { title, subtitle, image, buttonText, active } = req.body;
    
    // If setting as active, deactive others first
    if (active) {
        db.run("UPDATE hero SET active = 0");
    }

    db.run("INSERT INTO hero (title, subtitle, image, buttonText, active) VALUES (?, ?, ?, ?, ?)",
        [title, subtitle, image, buttonText, active ? 1 : 0], function(err) {
            if (err) return res.status(500).json({ message: 'Error adding hero content' });
            res.status(201).json({ id: this.lastID, ...req.body });
        });
});

app.put('/api/hero/:id', authenticateToken, authorizeAdmin, (req, res) => {
    const id = parseInt(req.params.id);
    const { title, subtitle, image, buttonText, active } = req.body;
    
    // Ensure active is explicitly treated as 0 or 1
    const activeValue = active ? 1 : 0;

    if (activeValue === 1) {
        // If we are setting this hero as active, first deactivate all others
        db.run("UPDATE hero SET active = 0", function(err) {
            if(err) {
                console.error("Error resetting active hero", err);
                return res.status(500).json({ message: 'Database error' });
            }
             updateHero();
        });
    } else {
        updateHero();
    }

    function updateHero() {
        db.run(`UPDATE hero SET 
            title = COALESCE(?, title), 
            subtitle = COALESCE(?, subtitle), 
            image = COALESCE(?, image), 
            buttonText = COALESCE(?, buttonText),
            active = ?
            WHERE id = ?`,
            [title, subtitle, image, buttonText, activeValue, id], function(err) {
                if (err) return res.status(500).json({ message: 'Error updating hero content' });
                if (this.changes === 0) return res.status(404).json({ message: 'Hero content not found' });
                res.json({ id, ...req.body, active: activeValue });
            });
    }
});

app.delete('/api/hero/:id', authenticateToken, authorizeAdmin, (req, res) => {
    const id = parseInt(req.params.id);
    db.run("DELETE FROM hero WHERE id = ?", [id], function(err) {
        if (err) return res.status(500).json({ message: 'Error deleting hero content' });
        if (this.changes === 0) return res.status(404).json({ message: 'Hero content not found' });
        res.json({ message: 'Hero content deleted' });
    });
});

// Orders
app.post('/api/orders', authenticateToken, (req, res) => {
    const { items, total_amount, shipping_address, payment_method } = req.body;
    const user_id = req.user.id;

    if (!items || items.length === 0) {
        return res.status(400).json({ message: 'No items in order' });
    }

    // Start transaction (simplified)
    db.serialize(() => {
        db.run(`INSERT INTO orders (user_id, total_amount, shipping_address, payment_method, tracking_number) VALUES (?, ?, ?, ?, ?)`,
            [user_id, total_amount, shipping_address, payment_method, `TRK-${Date.now()}`], 
            function(err) {
                if (err) return res.status(500).json({ message: 'Error creating order' });
                const orderId = this.lastID;

                const stmt = db.prepare("INSERT INTO order_items (order_id, product_id, quantity, price, size, color) VALUES (?, ?, ?, ?, ?, ?)");
                items.forEach(item => {
                    stmt.run(orderId, item.id, item.quantity, item.price, item.selectedSize || null, item.selectedColor || null);
                });
                stmt.finalize();

                res.status(201).json({ message: 'Order created successfully', orderId });
            }
        );
    });
});

app.get('/api/orders', authenticateToken, (req, res) => {
    const user_id = req.user.id;
    
    // Get orders with items
    const sql = `
        SELECT 
            o.id, o.total_amount, o.status, o.created_at, o.tracking_number,
            json_group_array(
                json_object(
                    'product_name', p.name,
                    'quantity', oi.quantity,
                    'price', oi.price,
                    'image', p.image,
                    'size', oi.size,
                    'color', oi.color
                )
            ) as items
        FROM orders o
        JOIN order_items oi ON o.id = oi.order_id
        JOIN products p ON oi.product_id = p.id
        WHERE o.user_id = ?
        GROUP BY o.id
        ORDER BY o.created_at DESC
    `;

    db.all(sql, [user_id], (err, rows) => {
        if (err) return res.status(500).json({ message: 'Database error ' + err.message });
        // Parse JSON string from json_group_array
        const orders = rows.map(row => ({
            ...row,
            items: JSON.parse(row.items)
        }));
        res.json(orders);
    });
});

app.get('/api/admin/orders', authenticateToken, authorizeAdmin, (req, res) => {
    const sql = `
        SELECT 
            o.id, o.total_amount, o.status, o.created_at, o.tracking_number, u.name as user_name, u.email as user_email,
            json_group_array(
                json_object(
                    'product_name', p.name,
                    'quantity', oi.quantity,
                    'price', oi.price,
                    'size', oi.size,
                    'color', oi.color
                )
            ) as items
        FROM orders o
        JOIN users u ON o.user_id = u.id
        JOIN order_items oi ON o.id = oi.order_id
        JOIN products p ON oi.product_id = p.id
        GROUP BY o.id
        ORDER BY o.created_at DESC
    `;

    db.all(sql, (err, rows) => {
        if (err) return res.status(500).json({ message: 'Database error' });
        const orders = rows.map(row => ({
            ...row,
            items: JSON.parse(row.items)
        }));
        res.json(orders);
    });
});

// Handle React routing, return all requests to React app
if (process.env.NODE_ENV === 'production') {
    app.get('*', (req, res) => {
        res.sendFile(path.join(__dirname, 'public', 'index.html'));
    });
}

if (process.env.NODE_ENV !== 'test') {
    app.listen(port, () => {
        console.log(`Server running on port ${port}`);
    });
}

module.exports = app;
