const sqlite3 = require('sqlite3').verbose();
const path = require('path');

const dbPath = path.resolve(__dirname, 'database.sqlite');
const db = new sqlite3.Database(dbPath);

db.serialize(() => {
    // 1. Create Roles Table
    db.run(`CREATE TABLE IF NOT EXISTS roles (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT UNIQUE,
        description TEXT
    )`, (err) => {
        if (err) console.error("Error creating roles table:", err.message);
        else console.log("Roles table created.");
    });

    // Seed Roles
    const roles = [
        { name: 'admin', description: 'Administrator with full access' },
        { name: 'customer', description: 'Standard customer user' }
    ];
    
    roles.forEach(role => {
        db.run("INSERT OR IGNORE INTO roles (name, description) VALUES (?, ?)", [role.name, role.description], (err) => {
             if (err) console.error(`Error seeding role ${role.name}:`, err.message);
        });
    });

    // 2. Update Products Table (Add new columns)
    const newProductColumns = [
        "description TEXT",
        "specifications TEXT",
        "dimensions TEXT",
        "weight REAL",
        "materials TEXT",
        "care_instructions TEXT",
        "warranty_info TEXT",
        "sku TEXT UNIQUE",
        "barcode TEXT",
        "stock INTEGER DEFAULT 0",
        "pricing_tier TEXT"
    ];

    newProductColumns.forEach(col => {
        // Simple extraction of column name for checking existence
        const colName = col.split(' ')[0]; 
        db.run(`ALTER TABLE products ADD COLUMN ${col}`, (err) => {
            if (err && !err.message.includes("duplicate column")) {
                console.error(`Error adding column ${colName}:`, err.message);
            } else {
                console.log(`Column ${colName} added to products.`);
            }
        });
    });

    // 3. Create Orders Table
    db.run(`CREATE TABLE IF NOT EXISTS orders (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        total_amount REAL,
        status TEXT DEFAULT 'pending',
        shipping_address TEXT,
        payment_method TEXT,
        tracking_number TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users (id)
    )`, (err) => {
        if (err) console.error("Error creating orders table:", err.message);
        else console.log("Orders table created.");
    });

    // 4. Create Order Items Table
    db.run(`CREATE TABLE IF NOT EXISTS order_items (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        order_id INTEGER,
        product_id INTEGER,
        quantity INTEGER,
        price REAL,
        FOREIGN KEY (order_id) REFERENCES orders (id),
        FOREIGN KEY (product_id) REFERENCES products (id)
    )`, (err) => {
        if (err) console.error("Error creating order_items table:", err.message);
        else console.log("Order Items table created.");
    });
});

db.close((err) => {
    if (err) {
        console.error(err.message);
    }
    console.log('Database schema update complete.');
});
